<?php
    /**
    *
    * copyright (c) 2015-2024 Heliac Technologies
    * license http://opensource.org/licenses/gpl-license.php GNU GPL v3 
    * 
    * Classe per eseguire delle ricerche nel database, consente di cercare 
    * 
    * @param $db oggetto database, deve essere un'istanza della classe DbManager di gUtil
    * @param $parametri array contenente i parametri per effettuare la ricerca: ;
    *          tabella:           la tabella su cui effettuare la ricerca
    *          elementiPerPagina: il numero di elementi che contiene una pagina 
    *          parametriRicerca:  array associativo con i parametri da ricercare nel database, ad esempio
    *                             ad esempio array("titolo"=>"testoDaCercare") 
    *                             Nota che titolo deve essere una colonna della tabella in cui si esegue la ricerca 
    * @param $impostazioniRicerca array contenente i parametri generali della ricerca:
    *          stringaNessunRisultato: la stringa da far comparire quando non ci sono risultati
    *          ordinamentoBase:        stringa con cui ordinare i risultati, può essere lasciata vuota 
    *          numeroElementiPagina:   array contenente il numero di possibili elementi per pagina,
               stringaSelectNumeroElementiPagina: stringa da visualizzare se viene mostrata la possibilità di scegliere il numero di elementi per pagina
    *                                           esempio:  "Visualizza _SELECT_NUMERO_ELEMENTI_ elementi",
               stringaInfoElementi     stringa contenente le informazioni sulla vista attuale
    *                                  esempio: "Vista da _INIZIO_ a _FINE_ di _ELEMENTITOTALI_ elementi",
               stringaElaborazioneDati stringa che contiene cosa scrivere durante l'elaborazione, usata solo per ajax
    */
    
    class Ricerca {
        private $db;
        private $tabella;
        private $ordinamento;
        private $query;
        private $elementiPerPagina;
        private $parametriRicerca; //array associativo
        private $parametriDaEscludere; //array associativo
        private $queryAnnidate; //array associativo
        private $paginaCorrente;
        private $risultatoQueryRicerca;
        private $numeroOggettiTrovati;
        private $numeroPagine;
        private $impostazioniRicerca;
        private $ajax;
        private $linkPaginaRicerca;
        private $templateStampa;
        private $ricercaTerminiEsatti;
        private $condizionePrincipale;

        function __construct($db,$parametri,$impostazioniRicerca,$linkPaginaRicerca) {
            $this->db = $db;
            $this->tabella = $parametri["tabella"];
            $this->elementiPerPagina     = $parametri["elementiPerPagina"];
            $this->parametriRicerca      = $parametri["parametriRicerca"];
            $this->parametriDaEscludere  = [];
            $this->queryAnnidate         = [];
            $this->impostazioniRicerca   = $impostazioniRicerca;
            $this->ajax                  = $parametri["ajax"];
            $this->ordinamento           = $impostazioniRicerca["ordinamentoBase"];
            $this->query                 = "";
            $this->risultatoQueryRicerca = null;
            $this->numeroOggettiTrovati  = 0;
            $this->numeroPagine          = 0;
            $this->linkPaginaRicerca     = $linkPaginaRicerca;
            $this->ricercaTerminiEsatti  = false;
            $this->templateStampa        = (isset($parametri["templateStampa"])) ? $parametri["templateStampa"] : "";
            $this->condizionePrincipale  = "";
        }

        public function eseguiRicerca($pagina,$operatoreLogico = "AND",$ricercaTerminiEsatti = false, $distinguiLettereENumeri = true) {
            $this->eseguiQueryConteggio($operatoreLogico);
            $this->paginaCorrente = $pagina;
            if($this->paginaCorrente==1) {
                $stringaLimiteRisultati = "LIMIT $this->elementiPerPagina";
            }
            else {
                $elementiPrecedenti = ($this->paginaCorrente-1) * $this->elementiPerPagina;
                $stringaLimiteRisultati = "LIMIT $elementiPrecedenti, $this->elementiPerPagina";
            }

            $stringaParametriRicerca = $this->generaStringaParametriRicerca($operatoreLogico);
            
            if($this->condizionePrincipale!="" && count($this->parametriRicerca)>0) {
                $stringaParametriRicercaSenzaWhere = preg_replace("/WHERE/","",$stringaParametriRicerca,1); // viene rimosso solo il primo WHERE per evitare problemi con le query annidate
                $this->query = "SELECT * FROM {$this->tabella} WHERE $this->condizionePrincipale ($stringaParametriRicercaSenzaWhere)";
            }
            else if($this->condizionePrincipale!="" && count($this->parametriRicerca)==0) { // c'è solo una condizione principale senza altri parametri
                $this->query = "SELECT * FROM {$this->tabella} WHERE $this->condizionePrincipale";
            }
            else {         
                $this->query = "SELECT * FROM {$this->tabella} $stringaParametriRicerca";
            }
            
            if($this->ordinamento!="") {
                 $this->query =  $this->query." ".$this->ordinamento." ".$stringaLimiteRisultati;
            }
            else {
                 $this->query =  $this->query." ".$stringaLimiteRisultati;
            }
            
            $this->risultatoQueryRicerca = $this->db->eseguiQueryLettura($this->query);
            return $this->risultatoQueryRicerca;
        }

        /*
         * esegue la query per contare il numero totale di oggetti presenti della ricerca con i 
         * parametri inseriti. Il metodo imposta anche il numero di pagine dividendo il numero
         * di risultati trovati per il numero di elementi per pagina
         */
        public function eseguiQueryConteggio($operatoreLogico = "AND") {
            $stringaParametriRicerca = $this->generaStringaParametriRicerca($operatoreLogico);
           
            if($this->condizionePrincipale!="" && count($this->parametriRicerca)>0) {
                $stringaParametriRicercaSenzaWhere = preg_replace("/WHERE/","",$stringaParametriRicerca,1); // viene rimosso solo il primo WHERE per evitare problemi con le query annidate
                $this->query = "SELECT COUNT(*) FROM {$this->tabella} WHERE $this->condizionePrincipale ($stringaParametriRicercaSenzaWhere)";
            }
            else if($this->condizionePrincipale!="" && count($this->parametriRicerca)==0) { // c'è solo una condizione principale senza altri parametri
                $this->query = "SELECT * FROM {$this->tabella} WHERE $this->condizionePrincipale";
            }
            else {            
                $this->query = "SELECT COUNT(*) FROM {$this->tabella} $stringaParametriRicerca";
            }

            $this->risultatoQueryRicerca = $this->db->eseguiQueryLettura($this->query);
            $risultato = $this->risultatoQueryRicerca->fetch_array();
            $this->numeroOggettiTrovati = $risultato[0];
            $this->numeroPagine = ceil($this->numeroOggettiTrovati/$this->elementiPerPagina);
        }
        
        public function getNumeroPagine() {
            return ceil($this->numeroOggettiTrovati / $this->elementiPerPagina);
        }
        
        public function eseguiRicercaCasuale($pagina) {
            $query = "SELECT * FROM $this->tabella  ORDER BY RAND() LIMIT {$this->elementiPerPagina}";
            $this->risultatoQueryRicerca = $this->db->eseguiQueryLettura($query);
            $this->numeroOggettiTrovati = $this->risultatoQueryRicerca->num_rows;
            $this->numeroPagine = round($this->numeroOggettiTrovati/$this->elementiPerPagina);
        }
        
        public function stampaSelectNumeroElementi($cerca="",$parametriAggiuntivi=array()) {
            $operazioneOnClick = ($this->ajax==false) ? "onclick=\"document.formNumeroElementi.submit()\"" : "";
            echo"<form class=\"form-inline w-100\" name=\"formNumeroElementi\" action=\"$this->linkPaginaRicerca\" >
                <label>Mostra: </label>
                <input type=\"text\" class=\"d-none\" name=\"cerca\" value=\"$cerca\" >";
                foreach($parametriAggiuntivi as $parametro=>$valore) {
                    echo"<input type=\"text\" class=\"d-none\" name=\"$parametro\" value=\"$valore\">";
                }
                echo"<div class=\"input-group\">
                    <select class=\"custom-select ml-sm-2 mb-2 mb-sm-0\" name=\"elementiPerPagina\" id=\"inputGroupSelect03\" aria-label=\"Example select with button addon\">";
                    foreach($this->impostazioniRicerca["numeroElementiPagina"] as $numeroElementi) {
                        $selezionato = ($this->elementiPerPagina==$numeroElementi ? "selected" : "");
                        echo"<option value=\"$numeroElementi\" $selezionato>$numeroElementi</option>";
                    }
                echo"</select>
                    <div class=\"input-group-append\">
                        <button type=\"submit\" class=\"btn btn-primary\" $operazioneOnClick><i class=\"fas fa-sync-alt\"></i></button>
                    </div>
                </div>
            </form>";
        }
        
        /*
         * Consente di stampare un form per effettuare la ricerca con i parametri stabiliti
         * è possibile mettere dei parametri aggiuntivi che verranno passati come campi nascosti
         */
        public function stampaFormRicerca($cerca="",$parametriAggiuntivi=array()) {
            $operazioneOnchange = ($this->ajax==true) ? "onchange=\"document.formRicerca.submit()\"" : "";
            echo"<form class=\"form-inline\" name=\"formRicerca\" action=\"$this->linkPaginaRicerca\">
                    <div class=\"input-group\">";
                    foreach($parametriAggiuntivi as $parametro=>$valore) {
                        echo"<input type=\"text\" class=\"d-none\" name=\"$parametro\" value=\"$valore\">";
                    }
                    echo"<input type=\"text\" class=\"d-none\" name=\"elementiPerPagina\" value=\"$this->elementiPerPagina\">
                        <input type=\"text\" class=\"form-control\" id=\"cerca\" aria-describedby=\"cerca\" name=\"cerca\" placeholder=\"{$this->impostazioniRicerca["stringaRicerca"]}\" value=\"$cerca\" $operazioneOnchange>
                        <div class=\"input-group-append\">
                            <button type=\"submit\" class=\"btn btn-primary\"><i class=\"fas fa-search\"></i></button>
                        </div>
                    </div>
                </form>";
        }
        
        public function stampaRisultatiRicerca() {
            if($this->risultatoQueryRicerca->num_rows>0) {
                while($oggetto = $this->risultatoQueryRicerca->fetch_assoc()) {   
                    $this->stampaElementoCorrente($oggetto);
                }
            }
            else {
                echo $this->impostazioniRicerca["stringaNessunRisultato"];
            }
        }

        /*
         *  metodo che prende il template in input e sostituisce le occorrenze __nomecampo__ prendendo
         *  i nomi dei campi dall'array associativo restituito
         */
        public function stampaElementoCorrente($oggetto) {
            $rigaDaStampare = $this->templateStampa;
            foreach($oggetto as $nomeParametro=>$valore) {
                $rigaDaStampare = str_replace("__".$nomeParametro."__", $valore, $rigaDaStampare);
            }
            echo $rigaDaStampare;
        }

         /*
         * stampa la paginazione per la ricerca, ha bisogno di ricevere il link compreso
         * di tutti i parametri tranne ovviamente il numero di pagina
         */
        public function stampaPaginazione($mostraTestoRisultati,$linkPagina="",$numeroPagineMassimoDaVisualizzare = 25, $classiAggiuntive = "") {
            $link = ($linkPagina=="") ? $this->linkPaginaRicerca : $linkPagina;
            if(strpos($link, "?")>0) {
                $link.="&elementiPerPagina=$this->elementiPerPagina&pag=";
            }
            else {            
                $link.="?elementiPerPagina=$this->elementiPerPagina&pag=";
            }
            if($mostraTestoRisultati) {
                $this->stampaPaginazioneConTesto($link,$numeroPagineMassimoDaVisualizzare,$classiAggiuntive);
            }
            else {
                $this->stampaPaginazioneSenzaTesto($link,$numeroPagineMassimoDaVisualizzare,$classiAggiuntive);
            }
        }
        
        /*
         * stampa la paginazione per la ricerca, ha bisogno di ricevere il link compreso
         * di tutti i parametri tranne ovviamente il numero di pagina
         */
        public function stampaPaginazioneSenzaTesto($linkPagina="",$numeroPagineMassimoDaVisualizzare = 12, $classiAggiuntive = "") {
            echo"<div class=\"d-flex justify-content-between mt-3\">";
            if($this->numeroOggettiTrovati > $this->elementiPerPagina) {
                echo"<nav aria-label=\"Paginazione\">
                        <ul class=\"pagination $classiAggiuntive\">";
                $numeroPagine = ceil($this->numeroOggettiTrovati / $this->elementiPerPagina);
                // stampo freccia a sinistra per andare indietro di una pagina alla volta
                if($this->paginaCorrente>1) {
                    $prec = $this->paginaCorrente-1;
                    ?>
                        <li class="page-item">
				<a class="page-link" href="<?= $linkPagina ?>1" aria-label=\"Previous\">
					<i class="fas fa-angle-double-left pt-1"></i>
				</a>
                        </li>
                        <li class="page-item">
				<a class="page-link" href="<?= $linkPagina.$prec ?>" aria-label=\"Previous\">
					<i class="fas fa-angle-left pt-1"></i>
				</a>
                        </li>
                    <?php
                }
                
                if($numeroPagine<=$numeroPagineMassimoDaVisualizzare) {
                    for($i=1;$i<$numeroPagine+1;$i++) {
                        if($i==$this->paginaCorrente) {
                            echo"<li class=\"page-item active\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";				
                        }
                        else {
                            echo"<li class=\"page-item\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";
                        }
                    }
                }
                else {
                    if($this->paginaCorrente < $numeroPagineMassimoDaVisualizzare/2) {
                        $limiteInferiore = 1;
                        $limiteSuperiore = $numeroPagineMassimoDaVisualizzare;
                    }
                    else {
                        $limiteInferiore = 1 + floor($this->paginaCorrente) -floor($numeroPagineMassimoDaVisualizzare/2);
                        $limiteSuperiore = floor($numeroPagineMassimoDaVisualizzare/2) + $this->paginaCorrente;
                    }
                    
                    if($limiteSuperiore > $numeroPagine) {
                        $limiteSuperiore = $numeroPagine;
                    }
                    
                    if($limiteInferiore > 2) {
			$this->stampaFormVaiAllaPagina("Destra",$linkPagina);
                    }
                    for($i=$limiteInferiore;$i<$limiteSuperiore+1;$i++) {
                        if($i==$this->paginaCorrente) {
                            echo"<li class=\"page-item active\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";				
                        }
                        else {
                            echo"<li class=\"page-item\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";
                        }
                    }
                    if($limiteSuperiore < $numeroPagine) {
			$this->stampaFormVaiAllaPagina("Destra",$linkPagina);
                    }
                }

                // stampo freccia in fondo per andare avanti di una pagina alla volta
                if($this->paginaCorrente < $this->getNumeroPagine()) {
                    $succ = $this->paginaCorrente+1;
                    ?>
                         <li class="page-item">
				<a class="page-link" href="<?= $linkPagina.$succ ?>" aria-label="Next">
					<i class="fas fa-angle-right pt-1"></i>
				</a>
                         </li>
                         <li class="page-item">
				<a class="page-link" href="<?= $linkPagina.$this->getNumeroPagine() ?>" aria-label="Next">
					<i class="fas fa-angle-double-right pt-1"></i>
				</a>
                         </li>
                    <?php
                }
                echo"</ul>
                    </nav>";
            }
            echo"</div>";
        }
        
        /*
         * stampa la paginazione per la ricerca, ha bisogno di ricevere il link compreso
         * di tutti i parametri tranne ovviamente il numero di pagina
         */
        public function stampaPaginazioneConTesto($linkPagina="",$numeroPagineMassimoDaVisualizzare = 12, $classiAggiuntive = "") {
            echo"<div class=\"d-flex justify-content-between mt-3\">";
            $primoElementoPagina = ($this->paginaCorrente==1) ? 1 : ($this->paginaCorrente-1) * $this->elementiPerPagina +1;
            $stringaElementi = str_replace("_INIZIO_", $primoElementoPagina, $this->impostazioniRicerca["stringaInfoElementi"]);
            
            $ultimoElementoPagina = ($this->numeroOggettiTrovati < $this->paginaCorrente * $this->elementiPerPagina) ? $this->numeroOggettiTrovati : $this->paginaCorrente * $this->elementiPerPagina;
            $stringaElementi = str_replace("_FINE_", $ultimoElementoPagina, $stringaElementi);
            $stringaElementi = str_replace("_ELEMENTITOTALI_", $this->numeroOggettiTrovati, $stringaElementi);
            
            echo"<div>$stringaElementi</div>";
            if($this->numeroOggettiTrovati > $this->elementiPerPagina) {
                echo"<nav aria-label=\"Paginazione\">
                        <ul class=\"pagination $classiAggiuntive\">";
                $numeroPagine = ceil($this->numeroOggettiTrovati / $this->elementiPerPagina);
                
                // stampo freccia a sinistra per andare indietro di una pagina alla volta
                if($this->paginaCorrente>1) {
                    $prec = $this->paginaCorrente-1;
                    ?>
                        <li class="page-item">
				<a class="page-link" href="<?= $linkPagina ?>1" aria-label=\"Previous\">
					<i class="fas fa-angle-double-left pt-1"></i>
				</a>
                        </li>
                        <li class="page-item">
				<a class="page-link" href="<?= $linkPagina.$prec ?>" aria-label=\"Previous\">
					<i class="fas fa-angle-left pt-1"></i>
				</a>
                        </li>
                    <?php
                }
                
                if($numeroPagine<=$numeroPagineMassimoDaVisualizzare) {
		        for($i=1;$i<$numeroPagine+1;$i++) {
		            if($i==$this->paginaCorrente) {
		                echo"<li class=\"page-item active\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";				
		            }
		            else {
		                echo"<li class=\"page-item\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";
		            }
		        }
                }
                else {
                    if($this->paginaCorrente < $numeroPagineMassimoDaVisualizzare/2) {
                        $limiteInferiore = 1;
                        $limiteSuperiore = $numeroPagineMassimoDaVisualizzare;
                    }
                    else {
                        $limiteInferiore = 1 + floor($this->paginaCorrente) -floor($numeroPagineMassimoDaVisualizzare/2);
                        $limiteSuperiore = floor($numeroPagineMassimoDaVisualizzare/2) + $this->paginaCorrente;
                    }

                    if($limiteSuperiore > $numeroPagine) {
                        $limiteSuperiore = $numeroPagine;
                    }
                    
                    if($limiteInferiore > 2) {
			$this->stampaFormVaiAllaPagina("Sinistra",$linkPagina);
                    }
                    for($i=$limiteInferiore;$i<$limiteSuperiore+1;$i++) {
                        if($i==$this->paginaCorrente) {
                            echo"<li class=\"page-item active\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";				
                        }
                        else {
                            echo"<li class=\"page-item\"><a class=\"page-link\" href=\"$linkPagina"."$i\">$i</a></li>";
                        }
                    }
                    if($limiteSuperiore < $numeroPagine) {
			$this->stampaFormVaiAllaPagina("Destra",$linkPagina);
                    }
                }
                
                // stampo freccia in fondo per andare avanti di una pagina alla volta
                if($this->paginaCorrente < $this->getNumeroPagine()) {
                    $succ = $this->paginaCorrente+1;
                    ?>
                         <li class="page-item">
				<a class="page-link" href="<?= $linkPagina.$succ ?>" aria-label="Next">
					<i class="fas fa-angle-right pt-1"></i>
				</a>
                         </li>
                         <li class="page-item">
				<a class="page-link" href="<?= $linkPagina.$this->getNumeroPagine() ?>" aria-label="Next">
					<i class="fas fa-angle-double-right pt-1"></i>
				</a>
                         </li>
                    <?php
                }
                echo"</ul>
                    </nav>";
            }
            echo"</div>";
        }
        
        function stampaFormVaiAllaPagina($tipo,$linkPagina) {
            $parametriAggiuntivi = [];

            if(strpos($linkPagina,"?")>0) { // se il punto interrogativo è presente sono presenti dei parametri
                $arrayStr = explode("?",$linkPagina);
                $linkPagina   = $arrayStr[0];
                $arrParametri = explode("&",$arrayStr[1]);

                foreach($arrParametri as $parametro) {
                    $arrParametro = explode("=",$parametro);
                    if($arrParametro[0]!="pag") { // il parametro "pag" presente di default nel link non viene inserito in quanto viene messo dall'utente nel campo input apposito
                        $parametriAggiuntivi[$arrParametro[0]] = $arrParametro[1];
                    }
                }
            }
            ?>
            <div class="dropup">
                <li>
                    <a href="javascript:void(0)" class="page-link" dropdown" id="dropdownMenu<?= $tipo ?>" data-toggle="dropdown" aria-haspopup="true">...</a>
                    <div class="dropdown-menu p-2" aria-labelledby="dropdownMenu<?= $tipo ?>">
                        <form action="<?= $linkPagina ?>">
                            <?php
                                foreach($parametriAggiuntivi as $nome=>$valore) {
                                    ?>
                                    <input type="text" class="d-none" name="<?= $nome ?>" value="<?= $valore ?>">
                                    <?php
                                }
                            ?>
                            <div class="input-group">
                                <input type="number" min="1" class="form-control" name="pag" placeholder="Pagina" aria-label="Pagina">
                                <div class="input-group-append">
                                    <button type="submit" class="btn btn-primary">Vai</button>
                                </div>
                            </div>
                        </form>
                    </div>
                </li>
            </div>
            <?php
        }

        function generaStringaParametriRicerca($operatoreLogico = "AND") {
            
            $stringaParametriRicerca = (count($this->parametriRicerca)>0 || count($this->parametriDaEscludere)>0 || count($this->queryAnnidate)>0) ? "WHERE " : "";
            $i = 1;
            foreach($this->parametriRicerca as $nomeParametro=>$parametro) {
                
                $stringaParametro = $this->generaStringaParametro($nomeParametro,$parametro);

                if($i==1) {
                    $stringaParametriRicerca .="$stringaParametro "; 
                }
                else {
                    $stringaParametriRicerca .="$operatoreLogico $stringaParametro ";
                }
                $i++;

            }
            
            // aggiungo eventuali query annidate
            foreach($this->queryAnnidate as $nomeParametro=>$queryAnnidata) {
            	if($stringaParametriRicerca!="WHERE ") {
		        $stringaParametriRicerca .=" $operatoreLogico $nomeParametro $queryAnnidata "; 
            	}
            	else {
		        $stringaParametriRicerca .=" $nomeParametro $queryAnnidata "; 
            	}
            }
            
            // aggiungo i parametri da escludere, se ci sono. In tal caso, se ci sono anche dei parametri di ricerca 
            // la stringa dei parametri ricerca generata finora va messa tra parentesi
            if(count($this->parametriDaEscludere)>0 && count($this->parametriRicerca)>1) {
            	$stringaParametriRicerca = str_replace("WHERE","WHERE (",$stringaParametriRicerca).")";
            }
            
            $i = 1;
            foreach($this->parametriDaEscludere as $nomeParametro=>$parametro) {
                if(is_numeric($parametro)) { // se il parametro passato in input è un numero usa l'operatore =
                    $stringaParametro = "$nomeParametro !=$parametro";
                }
                else { //altrimenti è una stringa e usa LIKE
                    $stringaParametro =  "$nomeParametro  NOT LIKE '$parametro'";
                }
                
                // se sono stati inseriti dei parametri nel precedente ciclo va aggiunto l'operatore logico scelto
                if($i==1 && count($this->parametriRicerca)==0) {
                    $stringaParametriRicerca .= $stringaParametro; 
                }
                else {
                    $stringaParametriRicerca .="AND $stringaParametro ";
                }
                $i++;
            }
            
            return $stringaParametriRicerca;
        }
        
        /*
         * Metodo per geneare la stringa del singolo parametro passato in input, che può essere un numero, un array
         * o una stringa
         */
        function generaStringaParametro($nomeParametro,$parametro) {
            if(is_numeric($parametro)) { // se il parametro passato in input è un numero usa l'operatore =
                $stringaParametro = "$nomeParametro =$parametro";
            }
            else if(is_array($parametro)) { // se il parametro passato in input è un array viene creato un for in modo che il valore sia uno di questi inseriti
                $stringaParametro = "(";
                foreach($parametro as $singoloParametro) {
                    $stringaParametro .= "$nomeParametro LIKE '$singoloParametro' OR ";
                }
                $stringaParametro = substr($stringaParametro, 0, -3);
                $stringaParametro .= ")";
            }
            else if($parametro!="" && ($parametro[0]=="<" || $parametro[0]==">")) {
                $stringaParametro = "$nomeParametro $parametro";
            }
            else { //altrimenti è una stringa e usa LIKE
                $stringaParametro = ($this->ricercaTerminiEsatti==true) ? "$nomeParametro  LIKE '$parametro' " : "$nomeParametro  LIKE '%$parametro%' ";
            }
            return $stringaParametro;
        }
        
        //getter e setter
        function getDb() {
            return $this->db;
        }
        
        function getTabella() {
            return $this->tabella;
        }

        function getQuery() {
            return $this->query;
        }

        function getElementiPerPagina() {
            return $this->elementiPerPagina;
        }

        function getParametriRicerca() {
            return $this->parametriRicerca;
        }

        function getRisultatoQueryRicerca() {
            return $this->risultatoQueryRicerca;
        }

        function getNumeroOggettiTrovati() {
            return $this->numeroOggettiTrovati;
        }

        function getTemplateStampa() {
            return $this->templateStampa;
        }

        function getRicercaTerminiEsatti() {
            return $this->ricercaTerminiEsatti;
        }
        
        public function getCondizionePrincipale() {
            return $this->condizionePrincipale;
        }

        function setRicercaTerminiEsatti($ricercaTerminiEsatti) {
            $this->ricercaTerminiEsatti = $ricercaTerminiEsatti;
        }

        function setTabella($tabella) {
            $this->tabella = $tabella;
        }

        function setQuery($query) {
            $this->query = $query;
        }

        function getOrdinamento() {
            return $this->ordinamento;
        }

        function getPaginaCorrente() {
            return $this->paginaCorrente;
        }

        function setOrdinamento($ordinamento) {
            $this->ordinamento = $ordinamento;
        }

        function setPaginaCorrente($paginaCorrente) {
            $this->paginaCorrente = $paginaCorrente;
        }

        function setElementiPerPagina($elementiPerPagina) {
            $this->elementiPerPagina = $elementiPerPagina;
        }

        function setParametriRicerca($parametriRicerca) {
            $this->parametriRicerca = $parametriRicerca;
        }

        function setParametriDaEscludere($parametriDaEscludere) {
            $this->parametriDaEscludere = $parametriDaEscludere;
        }

        public function setQueryAnnidate($queryAnnidate): void {
            $this->queryAnnidate = $queryAnnidate;
        }

        function setRisultatoQueryRicerca($risultatoQueryRicerca) {
            $this->risultatoQueryRicerca = $risultatoQueryRicerca;
        }

        function setNumeroOggettiTrovati($numeroOggettiTrovati) {
            $this->numeroOggettiTrovati = $numeroOggettiTrovati;
        }

        function setTemplateStampa($templateStampa) {
            $this->templateStampa = $templateStampa;
        }
        
        public function setCondizionePrincipale($condizionePrincipale): void {
            $this->condizionePrincipale = $condizionePrincipale;
        }
    }
